/*
  Path node for use for Bombing Run which gives extended pathing to mappers.

  Author Mark Caldwell aka W@rHe@d of The Reliquary 
*/
class UTBRPathNode extends PathNode placeable;

/** if true, bot will not attack or defend from this node */
var() bool NotDefensivePosition;

struct RouteInfoStruct
{
    /** The final destination of this pathnode's route */
    var() NavigationPoint RouteTarget;
    
    /** Specifies the next node in the route */
    var() NavigationPoint MoveTarget;
    
    /** Description of path. Set by mapper. */
    var() string Description;
    
    /** 
    *Bot seeks this instead of RouteTarget. Useful for broken or difficult paths 
    *close to RouteTarget. Once reached, bot proceeds to RouteTarget. Bot uses nearest 
    *approach node. Path length is not checked.
    *Useful for when bot can't direcly seek RouteTarget and should rather go to some
    *other place from where it can reach it.         
    */    
    var() bool ApproachPath;
    
    /** 
    *Bot takes special consideration of this as a path. Bot takes nearest AssaultPath 
    *if it's horizontally closer to RouteTarget than bot, and is a shorter route.
    *Path should be weighted accordingly.    
    */    
    var() bool AssaultPath;

    /** Extra distance added to or subtracted from path length */
    var() float ExtraDistance;
        
    /** If specified, is absolute distance to MoveTarget (overrides calculated distance) */
    var() float DistanceToMoveTarget;
    
    /** If specified, is absolute distance to RouteTarget (overrides calculated distance) */
    var() float DistanceToRouteTarget;
    
    /** 
    *Percentage chance this route is used at all. 100% if unspecified.
    *Routes with ChanceOfUse set will be preferred over routes with ChanceOfUse not set.
    *One route with ChanceOfUse not set can be used as a default route, if desired.
    */
    var() float ChanceOfUse;
    
    /** If true, dynamic pathing is used to force bot to MoveTarget, otherwise normal pathing is used */
    var() bool ForceToMoveTarget;
       
    /** 
    *If true, bot can use this assault node even if bot is horizontally closer to RouteTarget.
    *Very useful for solving difficult pathing problems close to objectives which
    *require bot to use an assault path further away.
    *Be careful with this, as misusing it can make the bot keep looping back to
    *this assault path. This should never be a shortened path, and may need lengthening
    *to stop bot from using it unless needed.    
    */
    var() bool CanUseIfBotCloserToRouteTarget;     
    
    /** 
    *If set, bot considers this approach/assault node reached when within this radius and moves on to next node.
    *Great for adding pathing variety without pegging bot into one tight location.
    */
    var() float ReachedRadius;   
};

struct BRNodeInfoStruct
{
    var Actor Node;
    var RouteInfoStruct RouteInfo;
    var float Dist;
};

var() Array<RouteInfoStruct> RouteInfo;


function PostBeginPlay()
{
    local int i;
    local bool done;
    local RouteInfoStruct ri;
    
    //make approach paths by default move to RouteTarget
    for (i = 0; i < RouteInfo.Length; i++)
    {
        if (RouteInfo[i].ApproachPath && (RouteInfo[i].MoveTarget == none))
        {
            RouteInfo[i].MoveTarget = RouteInfo[i].RouteTarget;
        }    
    }

    //move RouteInfo's which have ChanceOfUse to top so they are evaluated first
        
    done = false;
   
    while (! done)
    {
        done = true;
        
        for (i = 1; i < RouteInfo.Length; i++)
        {
            if ((RouteInfo[i - 1].ChanceOfUse == 0) && (RouteInfo[i].ChanceOfUse != 0))
            {
                ri = RouteInfo[i];
                RouteInfo[i] = RouteInfo[i - 1];
                RouteInfo[i - 1] = ri;
                done = false;
            }
        }    
    }
}

//return RouteInfo for routeTarget. Return RouteInfo containing MoveTarget if getMoveTarget is true
function RouteInfoStruct GetRouteInfo(Actor routeTarget)
{
    local int i;
    local RouteInfoStruct ret;
       
    if (routeTarget == none)
    {
        return ret;
    }
       
    for (i = 0; i < RouteInfo.Length; i++)
    {
        if (RouteInfo[i].RouteTarget == routeTarget)
        {
            if (RouteInfo[i].ChanceOfUse > 0)
            {
                if ((Rand(100) + 1) > RouteInfo[i].ChanceOfUse)
                {
                    continue;
                } 
            }          
            
            return RouteInfo[i];
        }
    }  
    
    return ret;
}

//return length of path from node to ri.RouteTarget
function float GetPathDist(RouteInfoStruct ri)
{
    local float dist;
    local int cnt;
    local NavigationPoint node, routeTarget, nextNode;
    local RouteInfoStruct emptyInfo, nextNodeInfo;
    
    /*
       the path lengths are crudely added up using straight distances.
       unfortunately ut provides us with nothing to calculate the path dist from 
       node A to node B, it can only be done from the bot's current location to another
       node. They do have a UTScout class which is supposed to be able to do this but
       it has bugs. in testing found it often had failed paths when bot didn't and often
       had wrong distances. too unreliable to use.
    */
  
    node = self;
    routeTarget = ri.RouteTarget;
    
    while (node != none)
    {
        cnt++;
        if (cnt > 100)
        {
            //will happen for a circular path, which is not valid.
            LogInternal("************ WARNING: " $ self $ ".GetPathDist more than 100 nodes found in path. Probably circular path. Aborting.");
            break;
        }
        
        if (ri.DistanceToRouteTarget != 0)
        {
            dist += ri.DistanceToRouteTarget;
            break;
        }
        
        dist += ri.ExtraDistance;
        dist += node.ExtraCost;
        
        if (node == routeTarget)
        {
            nextNode = none;
        }
        else if (ri.MoveTarget != none)
        {
           nextNode = ri.MoveTarget;
           nextNodeInfo = UTBRPathNode(ri.MoveTarget).GetRouteInfo(routeTarget);           
        }
        else
        {
           nextNode = routeTarget;
           nextNodeInfo = emptyInfo;
        }
                
        if (nextNode != none)
        {
            if (ri.DistanceToMoveTarget != 0)
            {
                dist += ri.DistanceToMoveTarget;
            }
            else
            {
                dist += UTBRGame(WorldInfo.Game).DistanceBetween(node, nextNode);
            }
                                  
            node = nextNode;
            ri = nextNodeInfo;
            
            continue;            
        }
                
        node = none;
    }
  
    return dist;
}

//put bot on best entry point to path and return that place
function Actor FindPathEntryPoint(RouteInfoStruct mainRouteInfo, UTBRPlayerManager playerManager)
{
    local Actor currentNode, routeTarget;
    local float dist, dist2;
    local UTBot B;
    local int cnt;
    local bool useNextNode;
    local RouteInfoStruct emptyInfo, ri;
    local BRNodeInfoStruct nearestNodeInfo, nodeInfo, entryPointInfo, nextNodeInfo;
    local string s;
    
    entryPointInfo.Node = self;
    entryPointInfo.RouteInfo = mainRouteInfo;

    currentNode = self;
    routeTarget = mainRouteInfo.RouteTarget;
    ri = mainRouteInfo;
        
    B = playerManager.Bot;
    nearestNodeInfo.Dist = 999999999;
    cnt = 0;
    
    //find nearest node on this node's path and then compare that with this node.
    //we don't test the route distance on each node as an optimization since pathing
    //has some cost.
    while (currentNode != none)
    {       
        cnt++;
        if (cnt > 100)
        {
            //this could happen if path forms a circle which would be infinite
            LogInternal("WARNING: " $ self $ " FindPathEntryPoint runaway loop suspected. More than 100 nodes in path. Must have a circular path.");
            break;
        }

        nodeInfo.Node = currentNode;
        nodeInfo.Dist = playerManager.DistanceBetween(B.Pawn, currentNode);
        nodeInfo.RouteInfo = ri;
        
        if (nodeInfo.Dist < nearestNodeInfo.Dist)
        {
            nearestNodeInfo = nodeInfo;
        }

        if (ri.MoveTarget != none)
        {
            currentNode = ri.MoveTarget;            
            ri = emptyInfo;    
            if (UTBRPathNode(currentNode) != none)
            {
                ri = UTBRPathNode(currentNode).GetRouteInfo(routeTarget);
            }
        }
        else
        {
            currentNode = none;
        }
    }

    //set entry point to either path start node or the nearest node, whichever has 
    //shorter path to
    
    if (entryPointInfo.Node != nearestNodeInfo.Node)
    {
        dist = playerManager.RouteDist(entryPointInfo.Node);        
        dist2 = playerManager.RouteDist(nearestNodeInfo.Node);
        if (dist2 < dist)
        {
            entryPointInfo = nearestNodeInfo;
        }
    }
    
    //see which is better, the current entry point or the next node in the path
    //after current entry point.
    
    //if the bot is between current entry point and entry point's next node,
    //then consider making the next node the entry point.
    //even though the route to current entry point may be shorter than route to next
    //node, going directly to next node is an overall shorter route than going
    //to current entry point and then to next node.
    //
    //this is also very very important for failsafe to keep bot on route. 
    //it stops bot from continuously looping back to a node on a path, as it will
    //keep pushing bot to the next node on a path.
    //even though playerManager.PathSubTarget keeps bot on route, if bot aborted
    //route to make a short detour to get a weapon, then a new route must
    //be obtained, and we would not want a bot going backwards to a node it's
    //already visited.
                  
    nextNodeInfo.Node = entryPointInfo.RouteInfo.MoveTarget;
    if (UTBRPathNode(nextNodeInfo.Node) != none)
    {
        nextNodeInfo.RouteInfo = UTBRPathNode(nextNodeInfo.Node).GetRouteInfo(routeTarget);
    }

    if ((nextNodeInfo.Node == none) && (entryPointInfo.Node != routeTarget))
    {
        //very important to make sure routeTarget is considered 
        //even if not linked in path to see if bot should push
        //on to routeTarget or take path. example is if bot
        //is between routeTarget and end of path then bot should not
        //take path even if path dist is shorter(due to a shortened weight),
        //otherwise bot will just keep looping back to path
        nextNodeInfo.Node = routeTarget;      
    }
    
    if (playerManager.ReachedTarget(entryPointInfo.Node) && (entryPointInfo.RouteInfo.MoveTarget == none))
    {
        //already reached the end of the path
        entryPointInfo.Node = none;
        nextNodeInfo.Node = none;
    }
    
    if (nextNodeInfo.Node!= none)
    {
        dist = playerManager.RouteDist(entryPointInfo.Node);
        dist2 = playerManager.RouteDist(nextNodeInfo.Node);
        
        useNextNode = false;
        
        if (playerManager.ReachedTarget(entryPointInfo.Node))
        {
            useNextNode = true;
        }
        
        if ((dist2 < dist) && (nextNodeInfo.Node != routeTarget))
        {
            //if route dist to nextNode is shorter then always go to it.
            //don't do this test for routeTarget since it was already decided
            //to take this path rather than go directly to routeTarget
            useNextNode = true;
        }
        
        if (playerManager.IsBetween(entryPointInfo.Node, nextNodeInfo.Node))
        {
            //we do want this test if nextNode is routeTarget because if bot
            //is between path and routeTarget we should keep looping back to 
            //path but just push onward to routeTarget
            
            if (dist2 <= (playerManager.DistanceBetween(entryPointInfo.Node, nextNodeInfo.Node) + dist))
            {
                //even if path from entryPoint to nextNode is forced,
                //we can just go to nextNode if path to is short enough.
                //usually a forced path means some difficult pathing
                //exists between them so it's less likely that the path
                //to nextNode will be short enough.
                useNextNode = true;
            }
                       
            if (! entryPointInfo.RouteInfo.ForceToMoveTarget)
            {
                //if bot is allowed to freely path and between entryPoint and
                //nextNode, go to nextNode, as pathing to nextNode from bot's
                //current location is likely to be better than going to entryPoint
                //and then to nextNode.
                //if there is some pathing difficulty between them then mapper should
                //force the path using ForceToMoveTarget.
                useNextNode = true;
            }
        }
               
        if (dist2 >= UTBRGame(WorldInfo.Game).BAD_ROUTE_DIST)
        {
            //path broken to next node so don't go to it
            useNextNode = false;
        }            
                  
        if (useNextNode)
        {
            if (entryPointInfo.RouteInfo.MoveTarget == none)
            {
                //nextNode is routeTarget and it's better to move on to routeTarget, but
                //we should not force this because routeTarget wasn't a MoveTarget on the 
                //path, so return nothing and let normal pathing be done.
                //an example of this is a standalone assault node which has no movetarget
                //set, once reached bot should just normally path to routeTarget and
                //not go back to assault node.
                entryPointInfo.Node = none;
            }
            else
            {
                entryPointInfo = nextNodeInfo;            
            }
        }
    } 


    if (entryPointInfo.Node != none)
    {
        playerManager.SetPathSubTarget(entryPointInfo.Node);
        playerManager.BRNodeRouteInfo = entryPointInfo.RouteInfo;
        
        s = "Approach path ";            
        if (mainRouteInfo.AssaultPath)
        {
            s = "Assault path ";
        }

        playerManager.ShowPath("Using " $ s $ self $ " entry point " $ entryPointInfo.Node);
    }
    else
    {
        playerManager.ShowPath("No good entry point found for " $ self);
    } 
                    
    return entryPointInfo.Node;
}



defaultproperties
{
   //with this set, you can put these path nodes right in the air.
   //real helpful for guiding bots to just the right place, such as down a hole,
   //jump up in the air, etc.
   bNotBased=true
   
   Name="Default__UTBRPathNode"
   
   Begin Object Name=Sprite ObjName=Sprite Archetype=SpriteComponent'Engine.Default__SpriteComponent'
      Sprite=Texture2D'EngineResources.S_Route'
      HiddenGame=True
      AlwaysLoadOnClient=False
      AlwaysLoadOnServer=False
      Name="Sprite"
      ObjectArchetype=SpriteComponent'Engine.Default__SpriteComponent'
   End Object
   Components(0)=Sprite   
}
